package main

import (
	"3g-mdk-go/core"
	"encoding/json"
	"fmt"
	"github.com/google/uuid"
	"github.com/playwright-community/playwright-go"
	"golang.org/x/text/encoding/japanese"
	"golang.org/x/text/transform"
	"os"
	"path/filepath"
)

func main() {

	if len(os.Args) < 2 {
		fmt.Println("Specify the sample type.")
		fmt.Println("ex: mpi, card, cvs, paynowid")
		os.Exit(0)
	}

	switch os.Args[1] {
	case "mpi":
		mpi()
	case "card":
		card()
	case "paynowid":
		paynowid()
	case "cvs":
		cvs()
	case "bank":
		bank()
	case "fraud":
		fraud()
	case "amazonpay":
		amazonpay()
	case "carrier":
		carrier()
	case "paypay":
		paypay()
	case "rakuten":
		rakuten()
	case "alipayplus":
		alipayplus()
	default:
		fmt.Println("An invalid sample type was specified.")
		os.Exit(0)
	}

}

const (
	MerchantCcId         = "{Set Merchant CcId here.}"
	MerchantSecret       = "{Set Merchant Secret Key here.}"
	TokenApiKey          = "{Set Token API Key here.}"
	CardNumberVisa       = "4111111111111111"
	CardNumberMasterCard = "5555555555554444"
	CardNumberJcb        = "3528000000000007"
	ExpDate              = "12/50"
	Cvc                  = "1234"
	CcName               = "TEST TARO"
	LocalServerPort      = "8125"
	VISA                 = DummyCardType("VISA")
	MasterCard           = DummyCardType("MasterCard")
	JCB                  = DummyCardType("JCB")
	TempDirName          = "mdk-go-temp"
	Headless             = false
)

func getConfig() *core.MerchantConfig {
	config := core.NewMerchantConfig(MerchantCcId, MerchantSecret)
	config.DummyRequest = true
	return config
}

func getToken(dummyCardType DummyCardType) string {
	pw, browser, page := openBrowser()

	uuidObj, _ := uuid.NewRandom()
	dirName, _ := os.MkdirTemp("", TempDirName)
	defer func(path string) {
		err := os.RemoveAll(path)
		if err != nil {
			fmt.Printf("remove tempdir error %s\n", dirName)
		}
	}(dirName)

	fileName := filepath.Join(dirName, uuidObj.String()+".html")

	tmpFile, err := os.Create(fileName)
	if err != nil {
		fmt.Printf("create token html file error %s\n", fileName)
		os.Exit(-1)
	}

	_, err = tmpFile.Write([]byte(TokenHtml))
	if err != nil {
		fmt.Printf("write token html file error %s\n", fileName)
		os.Exit(-1)
	}
	err = tmpFile.Close()
	if err != nil {
		fmt.Printf("close token html file error %s\n", fileName)
		os.Exit(-1)
	}

	_, err = page.Goto("file://" + tmpFile.Name())
	if err != nil {
		fmt.Printf("could not open token html %s\n", tmpFile.Name())
		os.Exit(-1)
	}

	_ = page.Locator("#token_api_key").Fill(TokenApiKey)
	var cardNumber string
	switch dummyCardType {
	case VISA:
		cardNumber = CardNumberVisa
	case MasterCard:
		cardNumber = CardNumberMasterCard
	case JCB:
		cardNumber = CardNumberJcb
	default:
		cardNumber = CardNumberVisa
	}
	_ = page.Locator("#cardnumber").Fill(cardNumber)
	_ = page.Locator("#exp-date").Fill(ExpDate)
	_ = page.Locator("#cvc").Fill(Cvc)
	_ = page.Locator("#ccname").Fill(CcName)
	_ = page.Locator("#submit").Click(playwright.LocatorClickOptions{})
	_ = page.WaitForLoadState(playwright.PageWaitForLoadStateOptions{
		State: playwright.LoadStateNetworkidle,
	})

	content, _ := page.Locator("#result").TextContent()

	closeBrowser(pw, browser, page)

	var obj = &TokenResponse{}
	if err := json.Unmarshal([]byte(content), &obj); err != nil {
		fmt.Printf("could not unmarshal token response json %s\n", tmpFile.Name())
		os.Exit(-1)
	}

	if obj.Token == "" {
		fmt.Printf("could not get token: %s\n", obj.Message)
		os.Exit(-1)
	}

	return obj.Token
}

type html struct {
	responseContents string
	sjis             bool
}

func moveBrowser(html *html, pw *playwright.Playwright, browser playwright.Browser, page playwright.Page) {
	uuidObj, _ := uuid.NewRandom()
	dirName, _ := os.MkdirTemp("", TempDirName)
	defer func(path string) {
		err := os.RemoveAll(path)
		if err != nil {
			fmt.Printf("remove tempdir error %s\n", dirName)
		}
	}(dirName)

	fileName := filepath.Join(dirName, uuidObj.String()+".html")

	tmpFile, err := os.Create(fileName)
	if err != nil {
		fmt.Printf("create ResponseContents html file error %s\n", fileName)
		os.Exit(-1)
	}

	if html.sjis {
		e := japanese.ShiftJIS.NewEncoder()
		sjis, _, _ := transform.String(e, html.responseContents)
		_, err = tmpFile.Write([]byte(sjis))
	} else {
		_, err = tmpFile.Write([]byte(html.responseContents))
	}

	if err != nil {
		fmt.Printf("write ResponseContents html file error %s\n", fileName)
		os.Exit(-1)
	}
	err = tmpFile.Close()
	if err != nil {
		fmt.Printf("close ResponseContents html file error %s\n", fileName)
		os.Exit(-1)
	}

	_, err = page.Goto("file://" + tmpFile.Name())
	if err != nil {
		fmt.Printf("could not open ResponseContents html %s\n", tmpFile.Name())
		os.Exit(-1)
	}
}

func openAndMoveBrowser(html *html) (*playwright.Playwright, playwright.Browser, playwright.Page) {
	pw, browser, page := openBrowser()
	moveBrowser(html, pw, browser, page)
	return pw, browser, page
}

func openBrowser() (*playwright.Playwright, playwright.Browser, playwright.Page) {
	pw, err := playwright.Run()
	if err != nil {
		fmt.Printf("could not start playwright: %v", err)
		os.Exit(-1)
	}
	browser, err := pw.Chromium.Launch(
		playwright.BrowserTypeLaunchOptions{
			Headless: playwright.Bool(Headless),
		},
	)
	if err != nil {
		fmt.Printf("could not launch browser: %v", err)
		os.Exit(-1)
	}
	newContext, err := browser.NewContext(playwright.BrowserNewContextOptions{
		IgnoreHttpsErrors: playwright.Bool(true),
		AcceptDownloads:   playwright.Bool(true),
	})
	if err != nil {
		fmt.Printf("could not create a new browser context: %v", err)
		os.Exit(-1)
	}
	page, _ := newContext.NewPage()

	return pw, browser, page
}

func closeBrowser(pw *playwright.Playwright, browser playwright.Browser, page playwright.Page) {
	err := page.Close()
	if err != nil {
		fmt.Printf("could not close page: %v", err)
		os.Exit(-1)
	}
	err = browser.Close()
	if err != nil {
		fmt.Printf("could not close browser: %v", err)
		os.Exit(-1)
	}
	err = pw.Stop()
	if err != nil {
		fmt.Printf("could not stop playwright: %v", err)
		os.Exit(-1)
	}
}

type DummyCardType string

type TokenResponse struct {
	Token           string `json:"token"`
	TokenExpireDate string `json:"token_expire_date"`
	ReqCardNumber   string `json:"req_card_number"`
	Status          string `json:"status"`
	Code            string `json:"code"`
	Message         string `json:"message"`
}

const (
	TokenHtml = `<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>VeriTrans Token Sample</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.0.2/dist/css/bootstrap.min.css" rel="stylesheet"
          integrity="sha384-EVSTQN3/azprG1Anm3QDgpJLIm9Nao0Yz1ztcQTwFspd3yD65VohhpuuCOmLASjC" crossorigin="anonymous">
</head>
<body>

<div class="container pt-5">
    <h3>Request</h3>

    <form id="form" method="post">

        <div class="mb-3">
            <label class="form-label" for="token_api_key">トークンAPIキー</label>
            <input type="text" class="form-control" id="token_api_key" maxlength="50">
        </div>

        <div class="mb-3">
            <label class="form-label" for="cardnumber">カード番号</label>
            <input type="text" inputmode="numeric" pattern="\d*" class="form-control" id="cardnumber" maxlength="19">
        </div>

        <div class="mb-3">
            <label class="form-label" for="exp-date">カード有効期限</label>
            <input type="text" class="form-control" id="exp-date" maxlength="5">
        </div>

        <div class="mb-3">
            <label class="form-label" for="cvc">セキュリティコード</label>
            <input type="text" inputmode="numeric" pattern="\d*" class="form-control" id="cvc" maxlength="4">
        </div>

        <div class="mb-3">
            <label class="form-label" for="ccname">名義人</label>
            <input type="text" class="form-control" id="ccname" maxlength="60">
        </div>

        <div class="pt-3">
            <button type="button" id="submit" class="btn btn-primary">送信</button>
        </div>

    </form>
</div>
<div class="container pt-5">
    <h3>Response</h3>
    <table class="table table-bordered">
        <tr>
            <th scope="row" class="col-3"> HTTPステータスコード</th>
            <td class="col-9" id="http_status_code"></td>
        </tr>
        <tr>
            <th scope="row" class="col-3"> データ</th>
            <td class="col-9" id="result"></td>
        </tr>
    </table>
</div>

<script>
    window.onload = function () {
        let button = document.getElementById("submit");
        if (button) {
            button.addEventListener("click", function (e) {
                submitToken(e);
            })
        }
    }

    function submitToken(e) {
        let data = {};
        data.token_api_key = document.getElementById('token_api_key').value;
        if (document.getElementById('cardnumber')) {
            data.card_number = document.getElementById('cardnumber').value;
        }
        if (document.getElementById('exp-date')) {
            data.card_expire = document.getElementById('exp-date').value;
        }
        if (document.getElementById('cvc')) {
            data.security_code = document.getElementById('cvc').value;
        }
        if (document.getElementById('ccname') && document.getElementById('ccname').value !== "") {
            data.cardholder_name = document.getElementById('ccname').value;
        }

        let url = "https://api3.veritrans.co.jp/4gtoken";

        let xhr = new XMLHttpRequest();
        xhr.open('POST', url, true);
        xhr.setRequestHeader('Accept', 'application/json');
        xhr.setRequestHeader('Content-Type', 'application/json; charset=utf-8');
        xhr.withCredentials = true;
        xhr.addEventListener('loadend', function () {
            if (xhr.status === 0) {
                alert("error");
                return;
            }
            let response = JSON.parse(xhr.response);
            document.getElementById('cardnumber').value = "";
            document.getElementById('exp-date').value = "";
            document.getElementById('cvc').value = "";
            document.getElementById('ccname').value = "";
            document.getElementById('http_status_code').innerText = xhr.status;
            document.getElementById('result').innerText = xhr.response
        });
        xhr.send(JSON.stringify(data));
    }
</script>
</body>
</html>`
)
