package main

import (
	"3g-mdk-go/core"
	"3g-mdk-go/dto/Carrier"
	"3g-mdk-go/dto/Search"
	"context"
	"fmt"
	"github.com/google/uuid"
	"github.com/playwright-community/playwright-go"
	"log/slog"
	"net"
	"net/http"
	"net/http/httptest"
	"os"
	"time"
)

func carrier() {
	handler := http.HandlerFunc(
		func(w http.ResponseWriter, request *http.Request) {
			_, _ = fmt.Fprintln(w, "OK")
		},
	)
	listener, _ := net.Listen("tcp", ":"+LocalServerPort)
	server := httptest.Server{
		Listener: listener,
		Config:   &http.Server{Handler: handler},
	}
	server.Start()
	defer server.Close()

	config := getConfig()

	slog.SetDefault(
		slog.New(
			core.WithWriteTraceIdHandler(
				slog.NewTextHandler(os.Stdout, &slog.HandlerOptions{}),
			),
		),
	)

	redirectionUri := "http://localhost:" + LocalServerPort + "/"

	u, _ := uuid.NewRandom()
	ctx := core.WithTraceID(context.Background(), u.String())

	orderIdDocomo1 := "order-docomo-1-" + u.String()

	// CarrierAuthorize (docomo Test accountingType=0)
	carrierAuthorizeRequest := &Carrier.AuthorizeRequest{
		OrderId:           orderIdDocomo1,
		ServiceOptionType: "docomo",
		Amount:            "100",
		TerminalKind:      "1",
		ItemType:          "0",
		AccountingType:    "0",
		WithCapture:       "false",
		SuccessUrl:        redirectionUri,
		CancelUrl:         redirectionUri,
		ErrorUrl:          redirectionUri,
	}

	var carrierAuthorizeResponse Carrier.AuthorizeResponse
	core.Execute(carrierAuthorizeRequest, &carrierAuthorizeResponse, config, ctx)

	fmt.Printf("mstatus: %s\n", carrierAuthorizeResponse.Mstatus)
	fmt.Printf("vResultCode: %s\n", carrierAuthorizeResponse.VResultCode)

	if carrierAuthorizeResponse.ResponseContents == "" {
		fmt.Println("ResponseContents is empty")
		os.Exit(-1)
	}

	// It *MUST* be returned to the browser in the character code *Shift-JIS*. (DO)
	pw, browser, page := openAndMoveBrowser(&html{
		responseContents: carrierAuthorizeResponse.ResponseContents,
		sjis:             true,
	})

	_ = page.WaitForLoadState(playwright.PageWaitForLoadStateOptions{State: playwright.LoadStateDomcontentloaded})
	_ = page.Locator("input[type=\"submit\"]").Click()
	_ = page.GetByText("次へ").Click()
	_ = page.GetByText("承諾して購入する").Click()

	_ = page.Locator("input[type=\"submit\"]").Click()

	_ = page.WaitForURL(redirectionUri + "*")

	closeBrowser(pw, browser, page)

	u, _ = uuid.NewRandom()
	ctx = core.WithTraceID(context.Background(), u.String())

	// CarrierCapture
	captureRequest := &Carrier.CaptureRequest{
		OrderId:           orderIdDocomo1,
		ServiceOptionType: "docomo",
	}

	var captureResponse Carrier.CaptureResponse
	core.Execute(captureRequest, &captureResponse, config, ctx)

	fmt.Printf("mstatus: %s\n", captureResponse.Mstatus)
	fmt.Printf("vResultCode: %s\n", captureResponse.VResultCode)

	u, _ = uuid.NewRandom()
	ctx = core.WithTraceID(context.Background(), u.String())

	// Search
	searchRequest := &Search.Request{
		ServiceTypeCd:    []string{"carrier"},
		ContainDummyFlag: "true",
		SearchParameters: &Search.Parameters{
			Common: &Search.CommonSearchParameter{
				OrderId: orderIdDocomo1,
			},
		},
	}

	var searchResponse Search.Response
	core.Execute(searchRequest, &searchResponse, config, ctx)

	fmt.Printf("mstatus: %s\n", searchResponse.Mstatus)
	fmt.Printf("vResultCode: %s\n", searchResponse.VResultCode)
	fmt.Printf("searchCount: %v\n", searchResponse.SearchCount)
	fmt.Printf("overMaxCountFlag: %v\n", searchResponse.OverMaxCountFlag)

	u, _ = uuid.NewRandom()
	ctx = core.WithTraceID(context.Background(), u.String())

	orderIdDocomo2 := "order-docomo-2-" + u.String()

	// CarrierAuthorize (docomo Test accountingType=4)
	carrierAuthorizeRequest2 := &Carrier.AuthorizeRequest{
		OrderId:           orderIdDocomo2,
		ServiceOptionType: "docomo",
		TerminalKind:      "1",
		ItemType:          "0",
		AccountingType:    "4",
		SuccessUrl:        redirectionUri,
		CancelUrl:         redirectionUri,
		ErrorUrl:          redirectionUri,
	}

	var carrierAuthorizeResponse2 Carrier.AuthorizeResponse
	core.Execute(carrierAuthorizeRequest2, &carrierAuthorizeResponse2, config, ctx)

	fmt.Printf("mstatus: %s\n", carrierAuthorizeResponse2.Mstatus)
	fmt.Printf("vResultCode: %s\n", carrierAuthorizeResponse2.VResultCode)

	if carrierAuthorizeResponse2.ResponseContents == "" {
		fmt.Println("ResponseContents is empty")
		os.Exit(-1)
	}

	// It *MUST* be returned to the browser in the character code *Shift-JIS*. (DO)
	pw, browser, page = openAndMoveBrowser(&html{
		responseContents: carrierAuthorizeResponse2.ResponseContents,
		sjis:             true,
	})

	_ = page.WaitForLoadState(playwright.PageWaitForLoadStateOptions{State: playwright.LoadStateDomcontentloaded})
	_ = page.Locator("input[type=\"submit\"]").Click()
	_ = page.GetByText("次へ").Click()
	_ = page.GetByText("承諾して購入する").Click()

	_ = page.Locator("input[type=\"submit\"]").Click()

	_ = page.WaitForURL(redirectionUri + "*")

	closeBrowser(pw, browser, page)

	u, _ = uuid.NewRandom()
	ctx = core.WithTraceID(context.Background(), u.String())

	orderIdDocomo3 := "order-docomo-3-" + u.String()

	// CarrierReAuthorize (docomo)
	reAuthorizeRequest := &Carrier.ReAuthorizeRequest{
		OrderId:           orderIdDocomo3,
		ServiceOptionType: "docomo",
		OriginalOrderId:   orderIdDocomo2,
		WithCapture:       "false",
		Amount:            "100",
	}

	var reAuthorizeResponse Carrier.ReAuthorizeResponse
	core.Execute(reAuthorizeRequest, &reAuthorizeResponse, config, ctx)

	fmt.Printf("mstatus: %s\n", reAuthorizeResponse.Mstatus)
	fmt.Printf("vResultCode: %s\n", reAuthorizeResponse.VResultCode)

	u, _ = uuid.NewRandom()
	ctx = core.WithTraceID(context.Background(), u.String())

	orderIdAu := "order-au-" + u.String()

	// CarrierAuthorize (au Test accountingType=0)
	carrierAuthorizeRequest3 := &Carrier.AuthorizeRequest{
		OrderId:           orderIdAu,
		ServiceOptionType: "au",
		Amount:            "100",
		TerminalKind:      "1",
		ItemType:          "0",
		AccountingType:    "0",
		WithCapture:       "false",
		SuccessUrl:        redirectionUri,
		CancelUrl:         redirectionUri,
		ErrorUrl:          redirectionUri,
	}

	var carrierAuthorizeResponse3 Carrier.AuthorizeResponse
	core.Execute(carrierAuthorizeRequest3, &carrierAuthorizeResponse3, config, ctx)

	fmt.Printf("mstatus: %s\n", carrierAuthorizeResponse3.Mstatus)
	fmt.Printf("vResultCode: %s\n", carrierAuthorizeResponse3.VResultCode)

	if carrierAuthorizeResponse3.ResponseContents == "" {
		fmt.Printf("ResponseContents is empty")
		os.Exit(-1)
	}

	pw, browser, page = openAndMoveBrowser(&html{
		responseContents: carrierAuthorizeResponse3.ResponseContents,
		sjis:             false,
	})

	_ = page.WaitForLoadState(playwright.PageWaitForLoadStateOptions{State: playwright.LoadStateDomcontentloaded})
	_ = page.Locator("input[type=\"submit\"]").Click()
	_ = page.GetByText("購入する").Click()

	_ = page.WaitForURL(redirectionUri + "*")

	closeBrowser(pw, browser, page)

	u, _ = uuid.NewRandom()
	ctx = core.WithTraceID(context.Background(), u.String())

	// CarrierCancel
	cancelRequest := &Carrier.CancelRequest{
		OrderId:           orderIdAu,
		ServiceOptionType: "au",
	}

	var cancelResponse Carrier.CancelResponse
	core.Execute(cancelRequest, &cancelResponse, config, ctx)

	fmt.Printf("mstatus: %s\n", cancelResponse.Mstatus)
	fmt.Printf("vResultCode: %s\n", cancelResponse.VResultCode)

	u, _ = uuid.NewRandom()
	ctx = core.WithTraceID(context.Background(), u.String())

	orderIdSbKtai := "order-sb-ktai-" + u.String()

	now := time.Now()

	// CarrierAuthorize (sb_ktai Test accountingType=1)
	carrierAuthorizeRequest4 := &Carrier.AuthorizeRequest{
		OrderId:           orderIdSbKtai,
		ServiceOptionType: "sb_ktai", // ソフトバンクまとめて支払い（B）
		Amount:            "100",
		TerminalKind:      "1",
		ItemType:          "0",
		AccountingType:    "1",
		D3Flag:            "2",
		MpFirstDate:       now.Format("20060102"),
		MpDay:             "01",
		SuccessUrl:        redirectionUri,
		CancelUrl:         redirectionUri,
		ErrorUrl:          redirectionUri,
	}

	var carrierAuthorizeResponse4 Carrier.AuthorizeResponse
	core.Execute(carrierAuthorizeRequest4, &carrierAuthorizeResponse4, config, ctx)

	fmt.Printf("mstatus: %s\n", carrierAuthorizeResponse4.Mstatus)
	fmt.Printf("vResultCode: %s\n", carrierAuthorizeResponse4.VResultCode)

	if carrierAuthorizeResponse4.ResponseContents == "" {
		fmt.Printf("ResponseContents is empty")
		os.Exit(-1)
	}

	// It *MUST* be returned to the browser in the character code *Shift-JIS*. (SB)
	pw, browser, page = openAndMoveBrowser(&html{
		responseContents: carrierAuthorizeResponse4.ResponseContents,
		sjis:             true,
	})

	_ = page.WaitForLoadState(playwright.PageWaitForLoadStateOptions{State: playwright.LoadStateDomcontentloaded})
	_ = page.GetByText("同意").Click()
	_ = page.GetByText("支払う").Click()

	_ = page.Locator("input[type=\"submit\"]").Click()

	_ = page.WaitForURL(redirectionUri + "*")

	closeBrowser(pw, browser, page)

	u, _ = uuid.NewRandom()
	ctx = core.WithTraceID(context.Background(), u.String())

	// CarrierTerminate
	terminateRequest := &Carrier.TerminateRequest{
		OrderId:           orderIdSbKtai,
		ServiceOptionType: "sb_ktai",
	}

	var terminateResponse Carrier.TerminateResponse
	core.Execute(terminateRequest, &terminateResponse, config, ctx)

	fmt.Printf("mstatus: %s\n", terminateResponse.Mstatus)
	fmt.Printf("vResultCode: %s\n", terminateResponse.VResultCode)

	u, _ = uuid.NewRandom()
	ctx = core.WithTraceID(context.Background(), u.String())

	orderIdSbMatomete := "order-sb-matomete-" + u.String()

	// CarrierAuthorize (sb_matomete Test accountingType=1)
	carrierAuthorizeRequest5 := &Carrier.AuthorizeRequest{
		OrderId:           orderIdSbMatomete,
		ServiceOptionType: "sb_matomete", // ソフトバンクまとめて支払い（A）
		Amount:            "1050",
		TerminalKind:      "1",
		ItemType:          "0",
		AccountingType:    "1",
		ItemId:            "dummy-sc-100001050",
		SuccessUrl:        redirectionUri,
		CancelUrl:         redirectionUri,
		ErrorUrl:          redirectionUri,
	}

	var carrierAuthorizeResponse5 Carrier.AuthorizeResponse
	core.Execute(carrierAuthorizeRequest5, &carrierAuthorizeResponse5, config, ctx)

	fmt.Printf("mstatus: %s\n", carrierAuthorizeResponse5.Mstatus)
	fmt.Printf("vResultCode: %s\n", carrierAuthorizeResponse5.VResultCode)

	if carrierAuthorizeResponse5.ResponseContents == "" {
		fmt.Printf("ResponseContents is empty")
		os.Exit(-1)
	}

	pw, browser, page = openAndMoveBrowser(&html{
		responseContents: carrierAuthorizeResponse5.ResponseContents,
		sjis:             false,
	})

	_ = page.WaitForLoadState(playwright.PageWaitForLoadStateOptions{State: playwright.LoadStateDomcontentloaded})
	_ = page.Locator("input[type=\"submit\"]").Click()

	_ = page.Locator("input[type=\"submit\"][name=\"sofma.name_ok\"]").Click()

	_ = page.WaitForURL(redirectionUri + "*")

	closeBrowser(pw, browser, page)

	os.Exit(0)
}
