"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transaction = void 0;
const MerchantConfig_1 = require("./MerchantConfig");
const MdkLocalException_1 = require("./MdkLocalException");
const MdkUtil_1 = require("./MdkUtil");
const HttpClient_1 = require("./HttpClient");
const node_fetch_1 = require("node-fetch");
const crypto_1 = require("crypto");
class Transaction {
    _logger;
    _config;
    _httpClient;
    constructor(logger, config, httpClient) {
        this._logger = logger;
        this._config = config;
        this._httpClient = httpClient || new HttpClient_1.HttpClient(config);
    }
    async execute(requestDto, type) {
        try {
            this._logger.info("Transaction.execute() start");
            const url = Transaction.createSendUrl(requestDto, this._config.host, MerchantConfig_1.MerchantConfig.ADD_URL_PAYMENT, MerchantConfig_1.MerchantConfig.ADD_URL_PAYMENT_VERSION, MerchantConfig_1.MerchantConfig.PAYNOWID_SERVICE_TYPE, MerchantConfig_1.MerchantConfig.ADD_URL_VTID, MerchantConfig_1.MerchantConfig.ADD_URL_VTID_VERSION, MerchantConfig_1.MerchantConfig.SERVICE_COMMAND_SEARCH, MerchantConfig_1.MerchantConfig.SEARCH_SERVER, this._config.dummyRequest, MerchantConfig_1.MerchantConfig.DUMMY_SERVER);
            this._logger.info("========== connect url       ==> " + url);
            this._logger.info("========== user agent        ==> " + MerchantConfig_1.MerchantConfig.userAgent);
            if (this._config.mdkErrorMode) {
                this._logger.info("Transaction.execute() end");
                const exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.TYPE_A_INTERNAL_ERROR);
                return Transaction.getErrorResponse(exception.errorCode, exception.errorMessage, type);
            }
            const obj = MdkUtil_1.MdkUtil.convertDTOtoJsonObject(requestDto);
            const maskedCopy = JSON.parse(JSON.stringify(obj));
            MdkUtil_1.MdkUtil.maskJson(maskedCopy);
            const maskedLog = JSON.stringify(maskedCopy);
            this._logger.info("========== request mct data  ==> " + maskedLog);
            const bodyJson = Transaction.appendConnectParam(obj, this._config.merchantCcId, this._config.merchantSecretKey, this._config.dummyRequest, this._logger);
            requestDto.maskedLog = maskedLog;
            return this.sendRequest(url, bodyJson, type);
        }
        catch (e) {
            this._logger.info("Transaction.execute() end");
            if (e instanceof MdkLocalException_1.MdkLocalException) {
                return Transaction.getErrorResponse(e.errorCode, e.errorMessage, type);
            }
            else {
                const exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.TYPE_A_INTERNAL_ERROR, undefined, e);
                return Transaction.getErrorResponse(exception.errorCode, exception.errorMessage, type);
            }
        }
    }
    async sendRequest(url, bodyJson, type) {
        try {
            const response = await this._httpClient.execute(url, bodyJson);
            let responseDto;
            const body = await response.text();
            if (body != null) {
                responseDto = MdkUtil_1.MdkUtil.setResponseProperties(body, type);
            }
            else {
                const exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.TYPE_B_INTERNAL_ERROR);
                responseDto = Transaction.getErrorResponse(exception.errorCode, exception.errorMessage, type);
            }
            this._logger.info("========== response json     ==> " + MdkUtil_1.MdkUtil.deleteReturn(responseDto.resultJson));
            this._logger.info("Transaction.execute() end");
            return responseDto;
        }
        catch (e) {
            this._logger.info("Transaction.execute() end");
            let code = undefined;
            if (e.hasOwnProperty("code")) {
                code = e.code;
            }
            if (e instanceof node_fetch_1.FetchError) {
                let exception;
                switch (e.type) {
                    case "system":
                        if (code != null) {
                            if (code == "ESOCKETTIMEDOUT" || code == "ETIMEDOUT") {
                                exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.SERVER_TIMEOUT, undefined, e);
                            }
                            else if (code == "ECONNRESET" || code == "ECONNREFUSED") {
                                exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.SOCKET_IO_ERROR, undefined, e);
                            }
                            else if (code == "ENOTFOUND" || code == "ERR_DNS_SET_SERVERS_FAILED") {
                                exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.GW_CONNECT_ERROR, undefined, e);
                            }
                            else if (code.startsWith("ERR_TLS_")) {
                                exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.SOCKET_FACTORY_CREATION_ERROR, undefined, e);
                            }
                            else {
                                exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.TYPE_F_INTERNAL_ERROR, undefined, e);
                            }
                        }
                        else {
                            exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.TYPE_F_INTERNAL_ERROR, undefined, e);
                        }
                        break;
                    case "unsupported-redirect":
                    case "max-redirect":
                    case "no-redirect":
                        exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.TYPE_F_INTERNAL_ERROR, undefined, e);
                        break;
                    case "request-timeout":
                        exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.SERVER_TIMEOUT, undefined, e);
                        break;
                    case "body-timeout":
                        exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.SERVER_TIMEOUT, undefined, e);
                        break;
                    case "invalid-json":
                    case "max-size":
                        exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.BAD_GW, undefined, e);
                        break;
                    default:
                        exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.TYPE_F_INTERNAL_ERROR, undefined, e);
                }
                return Transaction.getErrorResponse(exception.errorCode, exception.errorMessage, type);
            }
            else if (e instanceof node_fetch_1.Response) {
                let exception;
                switch (e.status) {
                    case 500:
                        exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.INTERNAL_SERVER_ERROR);
                        break;
                    case 502:
                        exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.BAD_GW);
                        break;
                    case 503:
                        exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.SERVICE_UNAVAILABLE);
                        break;
                    default:
                        exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.GW_CONNECT_ERROR);
                }
                return Transaction.getErrorResponse(exception.errorCode, exception.errorMessage, type);
            }
            else if (e instanceof Error) {
                let exception;
                if (e.message.startsWith("SSL Error") || e.message.startsWith("Invalid protocol") ||
                    e.message.startsWith("Hostname/IP does not match certificate's altnames")) {
                    exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.SOCKET_FACTORY_CREATION_ERROR, undefined, e);
                }
                else {
                    exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.TYPE_F_INTERNAL_ERROR, undefined, e);
                }
                return Transaction.getErrorResponse(exception.errorCode, exception.errorMessage, type);
            }
            else {
                const exception = new MdkLocalException_1.MdkLocalException(this._logger, MdkLocalException_1.MdkLocalException.TYPE_B_INTERNAL_ERROR);
                return Transaction.getErrorResponse(exception.errorCode, exception.errorMessage, type);
            }
        }
    }
    static createSendUrl(requestDto, host, addUrlPayment, addUrlPaymentVersion, payNowIdServiceType, addUrlVtId, addUrlVtIdVersion, serviceCommandSearch, searchServer, dummyRequest, dummyServer) {
        const url = host;
        const serviceType = requestDto.serviceType;
        const serviceCommand = requestDto.serviceCommand;
        let serviceName = addUrlPayment;
        let version = addUrlPaymentVersion;
        if (payNowIdServiceType.some(value => value == serviceType)) {
            serviceName = addUrlVtId;
            version = addUrlVtIdVersion;
        }
        if (serviceName == addUrlPayment && serviceCommand == serviceCommandSearch) {
            serviceName = serviceName + "-" + searchServer;
        }
        if (dummyRequest == "1") {
            serviceName = dummyServer + "-" + serviceName;
        }
        return url + "/" + serviceName + "/" + version + "/" + serviceCommand + "/" + serviceType;
    }
    static appendConnectParam(requestDto, merchantCcId, merchantSecretKey, dummyRequest, logger) {
        requestDto.txnVersion = MerchantConfig_1.MerchantConfig.MDK_DTO_VERSION;
        requestDto.dummyRequest = dummyRequest;
        requestDto.merchantCcid = merchantCcId;
        const jsonValue = JSON.stringify(requestDto);
        const maskedCopy = JSON.parse(jsonValue);
        MdkUtil_1.MdkUtil.maskJson(maskedCopy);
        const maskedLog = JSON.stringify(maskedCopy);
        logger.info("========== request mdk data  ==> " + maskedLog);
        const sha256 = (0, crypto_1.createHash)("sha256");
        sha256.update(merchantCcId + jsonValue + merchantSecretKey);
        const authHash = sha256.digest('hex');
        const obj = {
            "params": requestDto,
            "authHash": authHash
        };
        return JSON.stringify(obj);
    }
    static getErrorResponse(vResultCode, errorMessage, type) {
        const responseDto = new type();
        responseDto.mstatus = "failure";
        if (vResultCode != null && vResultCode.length == 4) {
            vResultCode += "000000000000";
        }
        if (vResultCode != null) {
            responseDto.vResultCode = vResultCode;
        }
        if (errorMessage != null) {
            responseDto.merrMsg = errorMessage;
        }
        return responseDto;
    }
}
exports.Transaction = Transaction;
//# sourceMappingURL=Transaction.js.map