import {assert, describe, it} from "vitest";
import {MdkUtil} from "../src/tgMdk/MdkUtil.js";
import {CardAuthorizeRequestDto} from "../src/tgMdkDto/Card/CardAuthorizeRequestDto.js";
import {FraudDetectionRequestDto} from "../src/tgMdkDto/FraudDetection/FraudDetectionRequestDto.js";
import {FraudDetectionDeviceDto} from "../src/tgMdkDto/FraudDetection/FraudDetectionDeviceDto.js";
import {FraudDetectionHeaderDto} from "../src/tgMdkDto/FraudDetection/FraudDetectionHeaderDto.js";
import {FraudDetectionTransactionDto} from "../src/tgMdkDto/FraudDetection/FraudDetectionTransactionDto.js";
import {FraudDetectionContactDto} from "../src/tgMdkDto/FraudDetection/FraudDetectionContactDto.js";
import {FraudDetectionSessionDto} from "../src/tgMdkDto/FraudDetection/FraudDetectionSessionDto.js";
import {FraudDetectionMethodCardDto} from "../src/tgMdkDto/FraudDetection/FraudDetectionMethodCardDto.js";
import {FraudDetectionUserAccountDto} from "../src/tgMdkDto/FraudDetection/FraudDetectionUserAccountDto.js";
import {FraudDetectionOrderDto} from "../src/tgMdkDto/FraudDetection/FraudDetectionOrderDto.js";
import {FraudDetectionLineItemDto} from "../src/tgMdkDto/FraudDetection/FraudDetectionLineItemDto.js";
import {FraudDetectionShipmentDto} from "../src/tgMdkDto/FraudDetection/FraudDetectionShipmentDto.js";
import {FraudDetectionCostDto} from "../src/tgMdkDto/FraudDetection/FraudDetectionCostDto.js";

import {CardAuthorizeResponseDto} from "../src/tgMdkDto/Card/CardAuthorizeResponseDto.js";
import {MpiAuthorizeRequestDto} from "../src/tgMdkDto/Mpi/MpiAuthorizeRequestDto";
import {FraudDetectionV2RequestDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2RequestDto";
import {FraudDetectionV2BankAccountDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2BankAccountDto";
import {FraudDetectionV2RiskDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2RiskDto";
import {FraudDetectionV2ParametersDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2ParametersDto";
import {FraudDetectionV2CustomerDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2CustomerDto";
import {
    FraudDetectionV2BrowserFingerprintDto
} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2BrowserFingerprintDto";
import {FraudDetectionV2BillingDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2BillingDto";
import {FraudDetectionV2ShippingDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2ShippingDto";
import {FraudDetectionV2CorporateDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2CorporateDto";
import {FraudDetectionV2CartDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2CartDto";
import {FraudDetectionV2ItemDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2ItemDto";
import {FraudDetectionV2GiftCardDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2GiftCardDto";
import {FraudDetectionV2AirlineDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2AirlineDto";
import {FraudDetectionV2PassengerDto} from "../src/tgMdkDto/FraudDetectionV2/FraudDetectionV2PassengerDto";
import {AccountCopyRequestDto} from "../src/tgMdkDto/PayNowId/AccountCopyRequestDto";
import {CardInfoAddRequestDto} from "../src/tgMdkDto/PayNowId/CardInfoAddRequestDto";

describe('MdkUtil Test', () => {


    it("Mask test", () => {
            assert.equal(MdkUtil.getMaskedValue("test", "test"), "test");

            assert.equal(MdkUtil.getMaskedValue("cardNumber", "2342-3423-4234-2341"), "234234******2341");
            assert.equal(MdkUtil.getMaskedValue("cardNumber", "2342342342342341"), "234234******2341");
            assert.equal(MdkUtil.getMaskedValue("cardNumber", "378282246310005"), "378282*****0005");
            assert.equal(MdkUtil.getMaskedValue("cardNumber", "3782-822463-10005"), "378282*****0005");
            assert.equal(MdkUtil.getMaskedValue("cardNumber", "36666666666660"), "366666****6660");
            assert.equal(MdkUtil.getMaskedValue("cardNumber", "3666-666666-6660"), "366666****6660");
            assert.equal(MdkUtil.getMaskedValue("cardNumber", "1234567890"), "**********");
            assert.equal(MdkUtil.getMaskedValue("cardNumber", "12345678901"), "123456*8901");
            assert.equal(MdkUtil.getMaskedValue("CardNumber", "4111111111111111"), "411111******1111");
            assert.equal(MdkUtil.getMaskedValue("cardnumber", "4111111111111111"), "411111******1111");
            assert.equal(MdkUtil.getMaskedValue("card_number", "4111111111111111"), "4111111111111111");

            assert.equal(MdkUtil.getMaskedValue("mailaddr", "foo@example.com"), "***@example.com");
            assert.equal(MdkUtil.getMaskedValue("mailaddr", "f@example.com"), "*@example.com");
            assert.equal(MdkUtil.getMaskedValue("mailaddr", "@example.com"), "@example.com");
            assert.equal(MdkUtil.getMaskedValue("mailaddr", "@example@.com"), "@example@.com");
            assert.equal(MdkUtil.getMaskedValue("mailaddr", "foo.example.com"), "foo.example.com");
            assert.equal(MdkUtil.getMaskedValue("mailaddr", "foo@t.co"), "***@t.co");
            assert.equal(MdkUtil.getMaskedValue("mailaddr", "foo@c"), "***@c");
            assert.equal(MdkUtil.getMaskedValue("mailaddr", "1@c"), "*@c");
            assert.equal(MdkUtil.getMaskedValue("mailaddr", "@"), "@");
            assert.equal(MdkUtil.getMaskedValue("mailaddr", "1@"), "*@");

            assert.equal(MdkUtil.getMaskedValue("mailAddress", "foo@example.com"), "***@example.com");
            assert.equal(MdkUtil.getMaskedValue("Mailaddr", "foo@example.com"), "***@example.com");
            assert.equal(MdkUtil.getMaskedValue("mail_addr", "foo@example.com"), "foo@example.com");
        }
    );


    const request = new CardAuthorizeRequestDto();
    request.amount = "100";
    request.cardNumber = '4111111111111111';
    request.orderId = 'someorderid';
    request.accountId = 'tarou';
    request.cardExpire = '12/25';
    request.token = '43c23d46-f5dc-4527-a21b-0646d61db7b3';
    request.birthday = "0526";
    request.tel = "08000000000";
    request.firstKanaName = "ﾃｽﾄﾀﾛｳ";
    request.lastKanaName = "ﾃｽﾄｽｽﾞｷ";

    it("Serialize and mask CardAuthorizeRequestDto ", () => {
        const obj = MdkUtil.convertDTOtoJsonObject(request);
        MdkUtil.maskJson(obj);
        assert.equal(obj["cardNumber"], "411111******1111");
        assert.equal(obj["cardExpire"], "*****");
        assert.equal(obj["birthday"], "****");
        assert.equal(obj["tel"], "***********");
        assert.equal(obj["firstKanaName"], "******");
        assert.equal(obj["lastKanaName"], "*******");
        assert.equal(obj["orderId"], "someorderid");
    });

    it("Serialize and mask CardAuthorizeRequestDto 2", () => {
        const request2 = new CardAuthorizeRequestDto();
        request2.accountId = "account-test"
        request2.groupId = "group-test"
        request2.orderId = "order-test"
        request2.amount = "100"
        request2.cardNumber = "4111111111111111"
        request2.cardExpire = "12/23"
        request2.withCapture = "true"
        const obj = MdkUtil.convertDTOtoJsonObject(request2);
        MdkUtil.maskJson(obj);
        assert.equal(obj["payNowIdParam"]["accountParam"]["accountId"], "account-test");
        assert.equal(obj["payNowIdParam"]["accountParam"]["recurringChargeParam"]["groupId"], "group-test");
        assert.equal(obj["cardNumber"], "411111******1111");
        assert.equal(obj["cardExpire"], "*****");
    })

    it("Serialize and mask AccountCopyRequestDto", () => {
        const request = new AccountCopyRequestDto();
        request.accountId = "account-test" + "_copied";
        request.originalAccountId = "account-test";
        request.deleteOriginalAccountId = "1";
        request.addCardInfo = "1";
        request.defaultCardOnly = "1";
        request.cleaningConfig = "1";
        const obj = MdkUtil.convertDTOtoJsonObject(request);
        MdkUtil.maskJson(obj);
        assert.equal(obj["payNowIdParam"]["accountParam"]["accountId"], "account-test_copied");
        assert.equal(obj["payNowIdParam"]["accountParam"]["originalAccountId"], "account-test");
        assert.equal(obj["payNowIdParam"]["accountParam"]["accountBasicParam"]["deleteOriginalAccountId"], "1");
        assert.isNotTrue((<any>obj).hasOwnProperty("_serviceType"));
    })

    it("Serialize and mask CardInfoAddRequestDto", () => {
        const request = new CardInfoAddRequestDto();
        request.accountId = "account-test"
        request.token = "abcdef01-2345-6789-abcd-ef0123456789";
        const obj = MdkUtil.convertDTOtoJsonObject(request);
        console.log(JSON.stringify(obj))
        MdkUtil.maskJson(obj);
        assert.equal(obj["payNowIdParam"]["accountParam"]["accountId"], "account-test");
        assert.equal(obj["payNowIdParam"]["accountParam"]["cardParam"]["token"], "abcdef01-2345-6789-abcd-ef0123456789");
        assert.isNotTrue((<any>obj).hasOwnProperty("_serviceType"));
    })

    it("Serialize and mask CardAuthorizeRequestDto with FraudDetection params", () => {
        const fraud = new FraudDetectionRequestDto(
            new FraudDetectionDeviceDto(
                "192.168.1.1",
                [
                    new FraudDetectionHeaderDto("testName", "testValue")
                ]
            )
        );

        fraud.contacts = [
            new FraudDetectionContactDto("1", "テスト", "タロウ", "JP"),
            new FraudDetectionContactDto("2", "テスト", "ハナコ", "JP")
        ];

        fraud.userAccount = new FraudDetectionUserAccountDto();
        fraud.userAccount.id = "EC0000123";

        fraud.transaction = new FraudDetectionTransactionDto();
        fraud.transaction.payer = "1";
        fraud.transaction.methodCard = new FraudDetectionMethodCardDto();
        fraud.transaction.methodCard.cardHolderName = "TAROU TEST";

        fraud.session = new FraudDetectionSessionDto("kjlsd9234jd0jse902pjsdfu0a23");

        fraud.order = new FraudDetectionOrderDto();
        fraud.order.lineItems = [
            new FraudDetectionLineItemDto("item1"),
            new FraudDetectionLineItemDto("item2"),
        ];
        fraud.order.shipment = new FraudDetectionShipmentDto();
        fraud.order.shipment.recipient = "2";
        fraud.order.shipment.cost = new FraudDetectionCostDto("300");
        fraud.order.shipment.lineItems = ["item1", "item2"];

        request.fraudDetectionRequest = fraud;

        const obj = MdkUtil.convertDTOtoJsonObject(request);
        MdkUtil.maskJson(obj);

        assert.equal(obj["fraudDetectionRequest"]["contacts"][0]["firstName"], "***");
        assert.equal(obj["fraudDetectionRequest"]["contacts"][0]["lastName"], "***");
        assert.equal(obj["fraudDetectionRequest"]["contacts"][1]["firstName"], "***");
        assert.equal(obj["fraudDetectionRequest"]["contacts"][1]["lastName"], "***");


    });


    it("deSerialize CardAuthorizeResponseDto", () => {

        const resultJson = "{\n" +
            "  \"vResultCode\": \"A001000000000000\",\n" +
            "  \"custTxn\": \"30065281\",\n" +
            "  \"acquirerCode\": \"05\",\n" +
            "  \"cardTransactiontype\": \"a\",\n" +
            "  \"centerRequestDate\": \"20190527113315\",\n" +
            "  \"centerResponseDate\": \"20190527113315\",\n" +
            "  \"connectedCenterId\": \"jcn\",\n" +
            "  \"fraudDetectionResponse\": {\n" +
            "    \"agResponse\": {\n" +
            "      \"decision\": \"accept\",\n" +
            "      \"hitReasons\": [\n" +
            "        \"DUMMY-REASON\"\n" +
            "      ],\n" +
            "      \"hitRules\": [\n" +
            "        \"DUMMY-RULE\"\n" +
            "      ]\n" +
            "    },\n" +
            "    \"result\": \"accept\",\n" +
            "    \"service\": \"ag\"\n" +
            "  },\n" +
            "  \"gatewayRequestDate\": \"20190527113315\",\n" +
            "  \"gatewayResponseDate\": \"20190527113315\",\n" +
            "  \"loopback\": \"0\",\n" +
            "  \"pending\": \"0\",\n" +
            "  \"reqAcquirerCode\": \"05\",\n" +
            "  \"reqAmount\": \"100\",\n" +
            "  \"reqCardExpire\": \"*****\",\n" +
            "  \"reqCardNumber\": \"411111********11\",\n" +
            "  \"reqItemCode\": \"0990\",\n" +
            "  \"resActionCode\": \"000\",\n" +
            "  \"resAuthCode\": \"000000\",\n" +
            "  \"resCenterErrorCode\": \"   \",\n" +
            "  \"resReturnReferenceNumber\": \"012345678901\",\n" +
            "  \"marchTxn\": \"30065281\",\n" +
            "  \"merrMsg\": \"処理が成功しました。\",\n" +
            "  \"mstatus\": \"success\",\n" +
            "  \"optionResults\": [],\n" +
            "  \"orderId\": \"order-1558924394\",\n" +
            "  \"serviceType\": \"card\",\n" +
            "  \"txnVersion\": \"2.0.7\"\n" +
            "}";

        try {
            const obj = JSON.parse(resultJson);
            const response: CardAuthorizeResponseDto = <CardAuthorizeResponseDto>obj;

            assert.equal(response.orderId, "order-1558924394");
        } catch (e) {
            assert.fail();
        }


    });

    it("deleteReturn test", () => {
        assert.equal(MdkUtil.deleteReturn("test\r\nTest"), "testTest");
        assert.equal(MdkUtil.deleteReturn("test\nTest"), "testTest");
        assert.equal(MdkUtil.deleteReturn("test\r\nTest\n"), "testTest");
        assert.equal(MdkUtil.deleteReturn(null), "");
        assert.equal(MdkUtil.deleteReturn(""), "");

    });

    it("convertDTOtoJsonObject test ", () => {
        const request = new MpiAuthorizeRequestDto();
        request.serviceOptionType = "mpi-company";
        request.httpUserAgent = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/114.0.0.0 Safari/537.36 Edg/114.0.1823.37";
        request.httpAccept = "text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.7";
        request.deviceChannel = "02";
        request.redirectionUri = "https://example.com/redirect";
        request.pushUrl = "https://example.com/push"
        request.orderId = "some_order_id";
        request.amount = "100";
        request.token = "abcdef01-2345-6789-abcd-ef0123456789";
        request.accountId = "some_account_id";
        request.groupId = "Veritrans_Dummy_01";
        request.createDate = "20300101"

        request.freeKey = "free_key";
        request.memo1 = "memo_1";

        // #region fraud
        // region fraud
        const fraud = new FraudDetectionV2RequestDto();
        fraud.fraudDetectionMode = "3"; // dummy response
        fraud.bankAccount = new FraudDetectionV2BankAccountDto();
        fraud.bankAccount.holder = "テスト タロウ";
        fraud.risk = new FraudDetectionV2RiskDto();
        fraud.risk.merchantWebsite = "https://www.example.com";
        fraud.risk.parameters = new FraudDetectionV2ParametersDto();
        fraud.risk.parameters.timeOnFile = "163";
        fraud.customer = new FraudDetectionV2CustomerDto();
        fraud.customer.merchantCustomerId = "USER10001";
        fraud.customer.givenName = "タロウ";
        fraud.customer.surname = "テスト";
        fraud.customer.browserFingerprint = new FraudDetectionV2BrowserFingerprintDto();
        fraud.customer.browserFingerprint.value = "some value";
        fraud.customer.status = "EXISTING";
        fraud.billing = new FraudDetectionV2BillingDto();
        fraud.billing.postcode = "1500022";
        fraud.billing.country = "JP";
        fraud.billing.city = "東京都";
        fraud.billing.street1 = "渋谷区"
        fraud.billing.street2 = "恵比寿南3-5-7"
        fraud.billing.houseNumber1 = "アーバンハウス101";
        fraud.shipping = new FraudDetectionV2ShippingDto();
        fraud.shipping.customer = fraud.customer;
        fraud.shipping.postcode = "1500022";
        fraud.shipping.country = "JP";
        fraud.shipping.city = "東京都";
        fraud.shipping.street1 = "渋谷区"
        fraud.shipping.street2 = "恵比寿南3-5-7"
        fraud.shipping.houseNumber1 = "アーバンハウス101";
        fraud.corporate = new FraudDetectionV2CorporateDto();
        fraud.corporate.name = "DG Financial Technology"
        fraud.cart = new FraudDetectionV2CartDto();
        const item = new FraudDetectionV2ItemDto();
        item.quantity = "1";
        item.partNumber = "4912345678904";
        item.productCode = "101";
        item.sku = "10";
        item.merchantItemId = "z12345678";
        item.description = "カレンダー";
        item.originalPrice = "100";
        item.type = "P";
        fraud.cart.items = [item];
        fraud.giftCard = new FraudDetectionV2GiftCardDto();
        fraud.giftCard.message = "歳末のご挨拶を申し上げます。";
        fraud.airline = new FraudDetectionV2AirlineDto();
        fraud.airline.ticketIssueDate = "2022-06-10";
        const passenger = new FraudDetectionV2PassengerDto();
        passenger.email = "foo@example.com";
        passenger.name = "テスト タロウ";
        fraud.airline.passengers = [passenger];
        request.fraudDetectionV2Request = fraud;
        // endregion fraud
        // #endregion

        const object = MdkUtil.convertDTOtoJsonObject(request);

        assert.equal(object["serviceOptionType"], "mpi-company");
        assert.equal(object["orderId"], "some_order_id");
        assert.equal(object["amount"], "100");
        assert.equal(object["httpUserAgent"], "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/114.0.0.0 Safari/537.36 Edg/114.0.1823.37");
        assert.equal(object["deviceChannel"], "02");
        assert.equal(object["payNowIdParam"]["token"], "abcdef01-2345-6789-abcd-ef0123456789");
        assert.equal(object["payNowIdParam"]["accountParam"]["accountId"], "some_account_id");
        assert.equal(object["payNowIdParam"]["accountParam"]["accountBasicParam"]["createDate"], "20300101");
        assert.equal(object["payNowIdParam"]["accountParam"]["recurringChargeParam"]["groupId"], "Veritrans_Dummy_01");
        assert.equal(object["fraudDetectionV2Request"]["fraudDetectionMode"], "3");
        assert.equal(object["fraudDetectionV2Request"]["bankAccount"]["holder"], "テスト タロウ");
        assert.equal(object["fraudDetectionV2Request"]["risk"]["parameters"]["timeOnFile"], "163");
        assert.equal(object["fraudDetectionV2Request"]["shipping"]["customer"]["browserFingerprint"]["value"], "some value");
        assert.equal(object["fraudDetectionV2Request"]["cart"]["items"][0]["partNumber"], "4912345678904");
        assert.equal(object["fraudDetectionV2Request"]["airline"]["passengers"][0]["email"], "foo@example.com");

    });

    it("setResponseProperties test", () => {
        const responseBody = "{\"payNowIdResponse\":{\"account\":{\"accountId\":\"account-1686212591\",\"cardInfo\":[{\"cardExpire\":\"12/25\",\"cardId\":\"ABCDEFGHIJKLMN0123456789Z\",\"cardNumber\":\"411111******1111\",\"defaultCard\":\"1\"}]},\"message\":\"処理が成功しました。\",\"processId\":\"23427100\",\"status\":\"success\"},\"result\":{\"vResultCode\":\"A001X00100000000\",\"custTxn\":\"8571910\",\"acquirerCode\":\"01\",\"cardTransactiontype\":\"a\",\"centerRequestDate\":\"20230608172316\",\"centerResponseDate\":\"20230608172316\",\"connectedCenterId\":\"jcn\",\"fraudDetectionResponse\":{\"rdResponse\":{\"code\":\"000.000.000\",\"orderId\":\"order1-1686212591\",\"requestId\":\"2023060805231609\",\"riskFraudStatusCode\":\"ACCEPT\",\"riskNeuralScore\":\"100\",\"riskOrderId\":\"20230608052316099\",\"riskResponseCode\":\"0100\",\"riskStatusCode\":\"PENDING\"},\"result\":\"accept\",\"service\":\"rd\"},\"gatewayRequestDate\":\"20230608172311\",\"gatewayResponseDate\":\"20230608172316\",\"kindCode\":\"3\",\"loopback\":\"0\",\"pending\":\"0\",\"reqAcquirerCode\":\"01\",\"reqAmount\":\"100\",\"reqCardExpire\":\"*****\",\"reqCardNumber\":\"411111******1111\",\"reqItemCode\":\"0990\",\"resActionCode\":\"000\",\"resAuthCode\":\"000000\",\"resCenterErrorCode\":\"   \",\"resReturnReferenceNumber\":\"012345678901\",\"marchTxn\":\"8571910\",\"merrMsg\":\"処理が成功しました。\",\"mstatus\":\"success\",\"optionResults\":[],\"orderId\":\"order1-1686212591\",\"serviceType\":\"card\",\"txnVersion\":\"2.0.7\"}}";

        const responseDto = MdkUtil.setResponseProperties(responseBody, CardAuthorizeResponseDto);

        assert.equal(responseDto.orderId, "order1-1686212591");
        assert.equal(responseDto?.payNowIdResponse?.processId, "23427100");
        assert.equal(responseDto?.payNowIdResponse?.account?.accountId, "account-1686212591");
        assert.equal(responseDto?.payNowIdResponse?.account?.cardInfo?.at(0)?.cardId, "ABCDEFGHIJKLMN0123456789Z");
        assert.equal(responseDto?.fraudDetectionResponse?.result, "accept");
        assert.equal(responseDto?.fraudDetectionResponse?.rdResponse?.riskFraudStatusCode, "ACCEPT");

    });

});
