import {describe, it, assert} from "vitest";
import {MerchantConfig} from "../src/tgMdk/MerchantConfig.js";
import {Transaction} from "../src/tgMdk/Transaction.js";
import {CardAuthorizeRequestDto} from "../src/tgMdkDto/Card/CardAuthorizeRequestDto.js";
import * as winston from 'winston';
import {CardAuthorizeResponseDto} from "../src/tgMdkDto/Card/CardAuthorizeResponseDto.js";
import {Mock, It, Times} from "moq.ts";
import type {IHttpClient} from "../src/tgMdk/IHttpClient.js";
import {FetchError, Response} from "node-fetch";
import {SearchRequestDto} from "../src/tgMdkDto/Search/SearchRequestDto";
import {SearchParameters} from "../src/tgMdkDto/Search/SearchParameters";
import {CommonSearchParameter} from "../src/tgMdkDto/Search/CommonSearchParameter";

describe('Transaction Test', () => {

    const merchantCcId = "some_merchant_ccid";
    const merchantSecret = "some_merchant_secret_key";
    const logger = winston.createLogger({
        transports: [
            new winston.transports.Console({ format: winston.format.simple() })
        ],
    });

    it("Card url Test", () => {

        const request = new CardAuthorizeRequestDto();
        request.amount = "100";
        request.orderId = 'some_order_id';
        request.accountId = 'some_account_id';
        request.token = 'abcdef01-2345-6789-abcd-ef0123456789';
        request.birthday = "0526";
        request.tel = "1234";
        request.firstKanaName = "ﾃｽﾄﾀﾛｳ";
        request.lastKanaName = "ﾃｽﾄｽｽﾞｷ";

        const config = new MerchantConfig(merchantCcId, merchantSecret);

        let url = Transaction.createSendUrl(request, config.host, MerchantConfig.ADD_URL_PAYMENT,
            MerchantConfig.ADD_URL_PAYMENT_VERSION, MerchantConfig.PAYNOWID_SERVICE_TYPE,
            MerchantConfig.ADD_URL_VTID, MerchantConfig.ADD_URL_VTID_VERSION,
            MerchantConfig.SERVICE_COMMAND_SEARCH, MerchantConfig.SEARCH_SERVER,
            config.dummyRequest, MerchantConfig.DUMMY_SERVER);

        assert.equal(url, "https://api3.veritrans.co.jp:443/paynow/v2/Authorize/card");

        url = Transaction.createSendUrl(request, config.host, MerchantConfig.ADD_URL_PAYMENT,
            MerchantConfig.ADD_URL_PAYMENT_VERSION, MerchantConfig.PAYNOWID_SERVICE_TYPE,
            MerchantConfig.ADD_URL_VTID, MerchantConfig.ADD_URL_VTID_VERSION,
            MerchantConfig.SERVICE_COMMAND_SEARCH, MerchantConfig.SEARCH_SERVER,
            "1", MerchantConfig.DUMMY_SERVER);

        assert.equal(url, "https://api3.veritrans.co.jp:443/test-paynow/v2/Authorize/card");

    });

    it("Search url Test", () => {

        const request = new SearchRequestDto();
        request.serviceTypeCd = ["card"]
        request.containDummyFlag = "true";
        request.searchParameters = new SearchParameters(new CommonSearchParameter("some_order_id"));

        const config = new MerchantConfig(merchantCcId, merchantSecret);

        let url = Transaction.createSendUrl(request, config.host, MerchantConfig.ADD_URL_PAYMENT,
            MerchantConfig.ADD_URL_PAYMENT_VERSION, MerchantConfig.PAYNOWID_SERVICE_TYPE,
            MerchantConfig.ADD_URL_VTID, MerchantConfig.ADD_URL_VTID_VERSION,
            MerchantConfig.SERVICE_COMMAND_SEARCH, MerchantConfig.SEARCH_SERVER,
            config.dummyRequest, MerchantConfig.DUMMY_SERVER);

        assert.equal(url, "https://api3.veritrans.co.jp:443/paynow-searchbyid/v2/Search/search");

        url = Transaction.createSendUrl(request, config.host, MerchantConfig.ADD_URL_PAYMENT,
            MerchantConfig.ADD_URL_PAYMENT_VERSION, MerchantConfig.PAYNOWID_SERVICE_TYPE,
            MerchantConfig.ADD_URL_VTID, MerchantConfig.ADD_URL_VTID_VERSION,
            MerchantConfig.SERVICE_COMMAND_SEARCH, MerchantConfig.SEARCH_SERVER,
            "1", MerchantConfig.DUMMY_SERVER);

        assert.equal(url, "https://api3.veritrans.co.jp:443/test-paynow-searchbyid/v2/Search/search");

    });

    it("Card request Test success", async () => {
        const config = new MerchantConfig(merchantCcId, merchantSecret);

        const responseMock = new Mock<Response>();
        responseMock.setup(instance => instance.text()).returns(
            new Promise<string>(resolve => resolve('{"result":{"vResultCode":"A001000000000000","custTxn":"30446978","acquirerCode":"05","cardTransactiontype":"a","centerRequestDate":"20190531140107","centerResponseDate":"20190531140107","connectedCenterId":"jcn","fraudDetectionResponse":{"agResponse":{"decision":"accept","hitReasons":["DUMMY-REASON"],"hitRules":["DUMMY-RULE"]},"result":"accept","service":"ag"},"gatewayRequestDate":"20190531140107","gatewayResponseDate":"20190531140107","loopback":"0","pending":"0","reqAcquirerCode":"05","reqAmount":"100","reqCardExpire":"*****","reqCardNumber":"411111********11","reqItemCode":"0990","resActionCode":"000","resAuthCode":"000000","resCenterErrorCode":"   ","resReturnReferenceNumber":"012345678901","marchTxn":"30446978","merrMsg":"処理が成功しました。","mstatus":"success","optionResults":[],"orderId":"order-1559278058","serviceType":"card","txnVersion":"2.0.8"}}')));
        responseMock.setup(instance => instance.status).returns(200);

        const httpClientMock = new Mock<IHttpClient>();
        httpClientMock.setup(
            instance => instance.execute(
                It.Is<string>(value => value === "https://api3.veritrans.co.jp:443/paynow/v2/Authorize/card"),
                It.IsAny<string>())
        ).returns(
            new Promise<Response>(resolve => resolve(responseMock.object())));

        const transaction = new Transaction(logger, config, httpClientMock.object());
        const request = genDto();

        const responseDto = await transaction.execute(request, CardAuthorizeResponseDto);
        httpClientMock.verify(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>()), Times.Once());
        assert.equal(responseDto.mstatus, "success");
        assert.equal(responseDto.orderId, request.orderId);

    });

    it("Card request Test 500", async () => {
        const config = new MerchantConfig(merchantCcId, merchantSecret);

        const response = new Response(null, {status: 500});

        let httpClientMock: Mock<IHttpClient>;
        httpClientMock = new Mock<IHttpClient>();
        httpClientMock.setup(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>())).throws(response);

        const transaction = new Transaction(logger, config, httpClientMock.object());

        const request = genDto();

        const responseDto = await transaction.execute(request, CardAuthorizeResponseDto);
        httpClientMock.verify(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>()), Times.Once());
        assert.equal(responseDto.vResultCode, "MF05000000000000");
    });

    it("Card request Test 502", async () => {
        const config = new MerchantConfig(merchantCcId, merchantSecret);

        const response = new Response(null, {status: 502});

        let httpClientMock: Mock<IHttpClient>;
        httpClientMock = new Mock<IHttpClient>();
        httpClientMock.setup(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>())).throws(response);

        const transaction = new Transaction(logger, config, httpClientMock.object());

        const request = genDto();

        const responseDto = await transaction.execute(request, CardAuthorizeResponseDto);
        httpClientMock.verify(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>()), Times.Once());
        assert.equal(responseDto.vResultCode, "MF06000000000000");
    });

    it("Card request Test 503", async () => {
        const config = new MerchantConfig(merchantCcId, merchantSecret);

        const response = new Response(null, {status: 503});

        let httpClientMock: Mock<IHttpClient>;
        httpClientMock = new Mock<IHttpClient>();
        httpClientMock.setup(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>())).throws(response);

        const transaction = new Transaction(logger, config, httpClientMock.object());

        const request = genDto();

        const responseDto = await transaction.execute(request, CardAuthorizeResponseDto);
        httpClientMock.verify(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>()), Times.Once());
        assert.equal(responseDto.vResultCode, "MF07000000000000");
    });

    it("Card request Test 505", async () => {
        const config = new MerchantConfig(merchantCcId, merchantSecret);

        const response = new Response(null, {status: 505});

        let httpClientMock: Mock<IHttpClient>;
        httpClientMock = new Mock<IHttpClient>();
        httpClientMock.setup(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>())).throws(response);

        const transaction = new Transaction(logger, config, httpClientMock.object());

        const request = genDto();
        const responseDto = await transaction.execute(request, CardAuthorizeResponseDto);
        httpClientMock.verify(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>()), Times.Once());
        assert.equal(responseDto.vResultCode, "MF02000000000000");
    });

    it("Card request Test ETIMEDOUT", async () => {
        const config = new MerchantConfig(merchantCcId, merchantSecret);

        const error: Record<string, unknown> = {
            code: "ETIMEDOUT"
        }
        const response = new FetchError("", "system", error);

        let httpClientMock: Mock<IHttpClient>;
        httpClientMock = new Mock<IHttpClient>();
        httpClientMock.setup(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>())).throws(response);

        const transaction = new Transaction(logger, config, httpClientMock.object());

        const request = genDto();

        const responseDto = await transaction.execute(request, CardAuthorizeResponseDto);
        httpClientMock.verify(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>()), Times.Once());
        assert.equal(responseDto.vResultCode, "MF03000000000000");
    });

    it("Card request Test ENOTFOUND", async () => {
        const config = new MerchantConfig(merchantCcId, merchantSecret);

        const error: Record<string, unknown> = {
            code: "ENOTFOUND"
        }
        const response = new FetchError("", "system", error);

        let httpClientMock: Mock<IHttpClient>;
        httpClientMock = new Mock<IHttpClient>();
        httpClientMock.setup(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>())).throws(response);

        const transaction = new Transaction(logger, config, httpClientMock.object());

        const request = genDto();

        const responseDto = await transaction.execute(request, CardAuthorizeResponseDto);
        httpClientMock.verify(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>()), Times.Once());
        assert.equal(responseDto.vResultCode, "MF02000000000000");
    });

    it("Card request Test other code", async () => {
        const config = new MerchantConfig(merchantCcId, merchantSecret);

        const error: Record<string, unknown> = {
            code: "OTHER"
        }
        const response = new FetchError("", "system", error);

        let httpClientMock: Mock<IHttpClient>;
        httpClientMock = new Mock<IHttpClient>();
        httpClientMock.setup(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>())).throws(response);

        const transaction = new Transaction(logger, config, httpClientMock.object());

        const request = genDto();

        const responseDto = await transaction.execute(request, CardAuthorizeResponseDto);
        httpClientMock.verify(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>()), Times.Once());
        assert.equal(responseDto.vResultCode, "MF99000000000000");
    });

    it("Card request Test SSL Error", async () => {
        const config = new MerchantConfig(merchantCcId, merchantSecret);

        const response = new Error("SSL Error");

        let httpClientMock: Mock<IHttpClient>;
        httpClientMock = new Mock<IHttpClient>();
        httpClientMock.setup(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>())).throws(response);

        const transaction = new Transaction(logger, config, httpClientMock.object());

        const request = genDto();

        const responseDto = await transaction.execute(request, CardAuthorizeResponseDto);
        httpClientMock.verify(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>()), Times.Once());
        assert.equal(responseDto.vResultCode, "MB03000000000000");
        assert.equal(responseDto.merrMsg, "SslStream creation error has occurred.")
    });

    it("Card request Test other message", async () => {
        const config = new MerchantConfig(merchantCcId, merchantSecret);

        const response = new Error("Some Error Message");

        let httpClientMock: Mock<IHttpClient>;
        httpClientMock = new Mock<IHttpClient>();
        httpClientMock.setup(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>())).throws(response);

        const transaction = new Transaction(logger, config, httpClientMock.object());

        const request = genDto();

        const responseDto = await transaction.execute(request, CardAuthorizeResponseDto);
        httpClientMock.verify(instance => instance.execute(It.IsAny<string>(), It.IsAny<string>()), Times.Once());
        assert.equal(responseDto.vResultCode, "MF99000000000000");
        assert.equal(responseDto.merrMsg, "System internal error")
    });

    function genDto(): CardAuthorizeRequestDto {
        const request = new CardAuthorizeRequestDto();
        request.amount = "100";
        request.orderId = 'order-1559278058';
        request.token = 'abcdef01-2345-6789-abcd-ef0123456789';
        return request;
    }

});
